<?php
// get_ads.php (PDO) — GET/POST
// ✅ CORS + Helpers + Auth (Session أو user_id+user_token)
// ✅ يرجّع إعلانات نشطة وصالحة للعرض الآن (limit 1..10)
// ✅ لا يرجّع name
// ✅ bg_color فقط:
//    - إذا bg_color فارغ/غير صالح => DEFAULT_BG_COLOR
//    - إذا bg_color صالح => يُستخدم
// ✅ banner_image_url:
//    - نقبله فقط إذا كان رابط صورة مباشر (امتداد صورة) + (يدعم نسبي ويحوله لمطلق)
//    - لو كان رابط صفحة (ليس صورة) => نتعامل معه كأنه لا يوجد صورة
// ✅ عند وجود صورة: نرجع الصورة + (نرجع النص/الخلفية كاحتياط في حال فشل تحميل الصورة على الواجهة)
// ✅ click_url اختياري: يقبل absolute http/https أو relative ويُحوّل على أساس https://eazzybit.com

if (session_status() === PHP_SESSION_NONE) session_start();

/* ================= CORS ================= */
$frontendOrigin = 'https://eazzybit.com';
$reqOrigin = $_SERVER['HTTP_ORIGIN'] ?? '';
if ($reqOrigin === $frontendOrigin) {
  header("Access-Control-Allow-Origin: {$frontendOrigin}");
  header("Vary: Origin");
  header("Access-Control-Allow-Credentials: true");
}
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Max-Age: 600");
header("Content-Type: application/json; charset=UTF-8");

/* ================= DEBUG ================= */
$DEBUG_MODE = false; // اجعله true مؤقتاً عند التشخيص فقط
if ($DEBUG_MODE) { ini_set('display_errors', 1); error_reporting(E_ALL); }
else { ini_set('display_errors', 0); error_reporting(0); }

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }

/* ================= Helpers ================= */
function respond(bool $success, string $message, array $extra = [], int $status = 200): void {
  http_response_code($status);
  echo json_encode(
    array_merge(['success' => $success, 'message' => $message], $extra),
    JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES
  );
  exit;
}

function log_error(Throwable $e): void {
  $dir = __DIR__ . '/../../logs';
  if (!is_dir($dir)) @mkdir($dir, 0755, true);
  $file = $dir . '/api_errors_' . date('Y-m-d') . '.log';
  $line = "[" . date('Y-m-d H:i:s') . "] " . $e->getMessage() . " | " . $e->getFile() . ":" . $e->getLine() . PHP_EOL;
  @file_put_contents($file, $line, FILE_APPEND | LOCK_EX);
}

function safe_int($v, int $default = 0): int {
  if ($v === null || $v === '') return $default;
  return (int)$v;
}

function read_input(): array {
  $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
  if ($method === 'POST') {
    $raw  = file_get_contents('php://input') ?: '';
    $json = json_decode($raw, true);
    if (is_array($json)) return $json;
    return $_POST ?? [];
  }
  return $_GET ?? [];
}

function base_host_url(): string {
  $https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
  $scheme = $https ? 'https' : 'http';
  $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
  return $scheme . '://' . $host;
}

/* ================= URL Normalizers ================= */

// absolute http/https only
function normalize_abs_http_url($u): ?string {
  $u = trim((string)($u ?? ''));
  if ($u === '') return null;
  if (!preg_match('~^https?://~i', $u)) return null;
  return $u;
}

// allow relative -> make absolute based on current host (admin/api domain)
function normalize_maybe_relative_url($u): ?string {
  $u = trim((string)($u ?? ''));
  if ($u === '') return null;

  // absolute
  if (preg_match('~^https?://~i', $u)) return $u;

  // relative "/path" or "path"
  $u = ltrim($u, '/');
  return rtrim(base_host_url(), '/') . '/' . $u;
}

// click_url: absolute http/https OR relative -> based on https://eazzybit.com
function normalize_click_url($u): ?string {
  $u = trim((string)($u ?? ''));
  if ($u === '') return null;

  if (preg_match('~^https?://~i', $u)) return $u;

  // relative => https://eazzybit.com/...
  $u = ltrim($u, '/');
  return 'https://eazzybit.com/' . $u;
}

// accept only direct image URLs by extension
function is_probably_image_url(string $url): bool {
  $path = parse_url($url, PHP_URL_PATH);
  if (!$path) return false;
  $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
  $allowed = ['jpg','jpeg','png','webp','gif','svg'];
  return in_array($ext, $allowed, true);
}

/* ✅ تنظيف لون الخلفية */
function normalize_bg_color($c): ?string {
  $c = trim((string)($c ?? ''));
  if ($c === '') return null;

  // معالجة أخطاء شائعة مثل "f6c544#" => "f6c544"
  if (substr($c, -1) === '#') $c = rtrim($c, '#');
  $c = trim($c);

  if ($c === '') return null;
  if (mb_strlen($c, 'UTF-8') > 64) return null;

  // HEX: "f6c544" أو "#f6c544" أو 3/6/8 digits
  if (preg_match('/^#?[0-9a-fA-F]{3}$/', $c) ||
      preg_match('/^#?[0-9a-fA-F]{6}$/', $c) ||
      preg_match('/^#?[0-9a-fA-F]{8}$/', $c)) {
    return ($c[0] === '#') ? $c : ('#' . $c);
  }

  // rgb/rgba/var(...)
  if (preg_match('/^(rgb|rgba)\(/i', $c)) return $c;
  if (preg_match('/^var\(--[a-zA-Z0-9\-_]+\)$/', $c)) return $c;

  // أسماء ألوان CSS البسيطة
  if (preg_match('/^[a-zA-Z]{3,20}$/', $c)) return $c;

  return null;
}

/* ================= DB ================= */
require_once __DIR__ . '/../config.php';
if (!isset($conn) || !($conn instanceof PDO)) {
  respond(false, 'اتصال قاعدة البيانات غير متوفر.', [], 500);
}

/* ================= AUTH ================= */
function resolveUserId(PDO $conn, array $in): int {
  // 1) Session
  $sid = isset($_SESSION['user_id']) ? (int)$_SESSION['user_id'] : 0;
  if ($sid > 0) return $sid;

  // 2) user_id + user_token
  $uid = isset($in['user_id']) ? (int)$in['user_id'] : 0;
  $tok = trim((string)($in['user_token'] ?? ''));
  if ($uid <= 0 || $tok === '') return 0;

  $st = $conn->prepare("
    SELECT user_id
    FROM user_sessions
    WHERE user_id = :uid
      AND user_token = :tok
      AND is_online = 1
    LIMIT 1
  ");
  $st->execute([':uid' => $uid, ':tok' => $tok]);
  $row = $st->fetch(PDO::FETCH_ASSOC);

  return $row ? (int)$row['user_id'] : 0;
}

try {
  $in = read_input();

  $userId = resolveUserId($conn, $in);
  if ($userId <= 0) {
    respond(false, 'غير مصرح. الرجاء تسجيل الدخول.', ['unauthorized' => true], 401);
  }

  $DEFAULT_BG_COLOR = '#f6c544';

  $limit = safe_int($in['limit'] ?? 5, 5);
  if ($limit < 1) $limit = 1;
  if ($limit > 10) $limit = 10;
  $limitSql = (int)$limit;

  // ✅ سماح NULL في publish_at / expires_at (كما طلبت سابقاً) مع احترام publish_now/open_ended
  $sql = "
    SELECT
      id,
      content_text,
      bg_color,
      banner_image_url,
      click_url,
      publish_now,
      publish_at,
      open_ended,
      expires_at
    FROM ads
    WHERE is_active = 1
      AND (publish_now = 1 OR publish_at IS NULL OR publish_at <= NOW())
      AND (open_ended = 1 OR expires_at IS NULL OR expires_at > NOW())
    ORDER BY id DESC
    LIMIT {$limitSql}
  ";

  $stmt = $conn->prepare($sql);
  $stmt->execute();
  $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

  $items = [];

  foreach ($rows as $r) {
    $id = (int)($r['id'] ?? 0);
    if ($id <= 0) continue;

    $text = trim((string)($r['content_text'] ?? ''));
    if ($text === '') $text = null;

    // bg_color فقط: لو غير صالح/فارغ => الافتراضي
    $bgRaw = normalize_bg_color($r['bg_color'] ?? null);
    $bg = $bgRaw ?: $DEFAULT_BG_COLOR;

    // banner: نقبل نسبي/مطلق لكن "لا" نعتبره صورة إلا لو امتداد صورة
    $bannerCandidate = normalize_maybe_relative_url($r['banner_image_url'] ?? null);
    $banner = null;
    if ($bannerCandidate && is_probably_image_url($bannerCandidate)) {
      $banner = $bannerCandidate;
    }

    $click = normalize_click_url($r['click_url'] ?? null);

    // إذا لا يوجد صورة ولا يوجد نص => تجاهل
    if ($banner === null && $text === null) continue;

    // ✅ نرجع الصورة إن وجدت + نرجع النص/الخلفية كاحتياط إذا فشل تحميل الصورة في الواجهة
    $items[] = [
      'id'               => $id,
      'has_image'        => ($banner !== null),
      'banner_image_url' => $banner,
      'text'             => $text,
      'bg_color'         => $bg,
      'click_url'        => $click,
    ];
  }

  respond(true, 'تم جلب الإعلانات بنجاح.', [
    'user_id' => $userId,
    'count'   => count($items),
    'items'   => $items,
  ]);

} catch (Throwable $e) {
  log_error($e);
  $msg = 'خطأ في الخادم';
  if ($DEBUG_MODE) $msg .= ' - ' . $e->getMessage();
  respond(false, $msg, [], 500);
}
