<?php
session_start();

/**
 * get_login.php (PDO) - FINAL + reCAPTCHA v3 (Server-side)
 * ✅ تسجيل الدخول بالبريد أو رقم الحساب أو اسم المستخدم
 * ✅ حماية من محاولات البوت عبر reCAPTCHA (Server-side verify)
 * ✅ يعيد needs_verification إذا كان الحساب غير مُفعّل
 * ✅ يولّد user_token ويخزّنه في user_tokens إذا كان الجدول موجوداً (بشكل مرن)
 * ✅ يضع $_SESSION['user_id'] عند نجاح تسجيل الدخول
 */

$frontendOrigin = 'https://eazzybit.com';
$reqOrigin = $_SERVER['HTTP_ORIGIN'] ?? '';
if ($reqOrigin === $frontendOrigin) {
  header("Access-Control-Allow-Origin: {$frontendOrigin}");
  header("Vary: Origin");
  header("Access-Control-Allow-Credentials: true");
}
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Max-Age: 600");
header("Content-Type: application/json; charset=UTF-8");

// وضع تطوير
$DEBUG_MODE = true;
if ($DEBUG_MODE) { ini_set('display_errors', 1); error_reporting(E_ALL); }
else { ini_set('display_errors', 0); error_reporting(0); }

// Preflight
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') {
  http_response_code(204);
  exit;
}

// POST فقط
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success' => false, 'message' => 'طريقة الطلب غير مسموحة. استخدم POST فقط.'], JSON_UNESCAPED_UNICODE);
  exit;
}

/* ================= أدوات مساعدة ================= */
function respond(bool $success, string $message, array $extra = [], int $status = 200): void {
  http_response_code($status);
  echo json_encode(
    array_merge(['success' => $success, 'message' => $message], $extra),
    JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES
  );
  exit;
}

function read_json_input(): array {
  $raw = file_get_contents('php://input') ?: '';
  $data = json_decode($raw, true);
  if (!is_array($data)) {
    respond(false, 'البيانات المرسلة غير صالحة', [], 400);
  }
  return $data;
}

function log_error(Throwable $e): void {
  $dir = __DIR__ . '/../../logs';
  if (!is_dir($dir)) @mkdir($dir, 0755, true);
  $file = $dir . '/api_errors_' . date('Y-m-d') . '.log';
  $line = "[" . date('Y-m-d H:i:s') . "] " . $e->getMessage() . " | " . $e->getFile() . ":" . $e->getLine() . PHP_EOL;
  @file_put_contents($file, $line, FILE_APPEND | LOCK_EX);
}

/* ================= reCAPTCHA (Server-side verify) ================= */
/**
 * يتحقق من reCAPTCHA ويدعم fallback إلى recaptcha.net عند فشل الوصول لـ google.com
 * - في v3: يمكن فحص score (إذا RECAPTCHA_MIN_SCORE معرّف)
 * - ويمكن فحص action (إذا وصل من Google) لتطابق "login"
 */
function verify_recaptcha(string $token, string $remoteIp = '', string $expectedAction = 'login'): array {
  if (!defined('RECAPTCHA_SECRET') || !RECAPTCHA_SECRET) {
    return ['ok' => false, 'reason' => 'RECAPTCHA_SECRET_NOT_SET'];
  }

  $endpoints = [
    'https://www.google.com/recaptcha/api/siteverify',
    'https://www.recaptcha.net/recaptcha/api/siteverify',
  ];

  $post = [
    'secret'   => RECAPTCHA_SECRET,
    'response' => $token,
  ];
  if ($remoteIp !== '') $post['remoteip'] = $remoteIp;

  $lastReason = 'VERIFY_REQUEST_FAILED';

  foreach ($endpoints as $url) {
    $respRaw = null;

    if (function_exists('curl_init')) {
      $ch = curl_init($url);
      curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($post),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_CONNECTTIMEOUT => 6,
        CURLOPT_TIMEOUT => 10,
      ]);
      $respRaw = curl_exec($ch);
      curl_close($ch);
    } else {
      $context = stream_context_create([
        'http' => [
          'method'  => 'POST',
          'header'  => "Content-Type: application/x-www-form-urlencoded\r\n",
          'content' => http_build_query($post),
          'timeout' => 10,
        ]
      ]);
      $respRaw = @file_get_contents($url, false, $context);
    }

    if (!$respRaw) {
      $lastReason = 'VERIFY_REQUEST_FAILED';
      continue;
    }

    $json = json_decode($respRaw, true);
    if (!is_array($json)) {
      $lastReason = 'VERIFY_BAD_RESPONSE';
      continue;
    }

    if (empty($json['success'])) {
      $lastReason = 'VERIFY_NOT_SUCCESS';
      continue;
    }

    // v3 action check (إذا موجود)
    if (isset($json['action']) && $expectedAction !== '') {
      if (strtolower((string)$json['action']) !== strtolower($expectedAction)) {
        return ['ok' => false, 'reason' => 'ACTION_MISMATCH', 'action' => (string)$json['action']];
      }
    }

    // v3 score (اختياري)
    if (isset($json['score']) && defined('RECAPTCHA_MIN_SCORE')) {
      $min = (float)RECAPTCHA_MIN_SCORE;
      $score = (float)$json['score'];
      if ($score < $min) {
        return ['ok' => false, 'reason' => 'LOW_SCORE', 'score' => $score];
      }
    }

    return ['ok' => true, 'data' => $json, 'endpoint' => $url];
  }

  return ['ok' => false, 'reason' => $lastReason];
}

/* ================= Token storage (مرن) ================= */
function table_exists(PDO $conn, string $table): bool {
  $q = $conn->prepare("SELECT 1 FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = ? LIMIT 1");
  $q->execute([$table]);
  return (bool)$q->fetchColumn();
}

function table_columns(PDO $conn, string $table): array {
  $q = $conn->prepare("
    SELECT column_name
    FROM information_schema.columns
    WHERE table_schema = DATABASE() AND table_name = ?
  ");
  $q->execute([$table]);
  $cols = [];
  foreach ($q->fetchAll(PDO::FETCH_COLUMN) as $c) {
    $cols[strtolower((string)$c)] = true;
  }
  return $cols;
}

function store_user_token(PDO $conn, int $userId, string $token, string $ip, string $ua): void {
  try {
    if (!table_exists($conn, 'user_tokens')) return;

    $cols = table_columns($conn, 'user_tokens');

    // لازم user_id
    if (empty($cols['user_id'])) return;

    // ابحث عن عمود التوكن
    $tokenCol = null;
    foreach (['user_token', 'token', 'access_token', 'auth_token'] as $c) {
      if (!empty($cols[$c])) { $tokenCol = $c; break; }
    }
    if (!$tokenCol) return;

    $insertCols = ['user_id', $tokenCol];
    $valuesSql  = ['?', '?'];
    $params     = [$userId, $token];

    // created_at
    if (!empty($cols['created_at'])) {
      $insertCols[] = 'created_at';
      $valuesSql[]  = 'NOW()';
    }

    // expires_at (30 يوم افتراضي)
    if (!empty($cols['expires_at'])) {
      $insertCols[] = 'expires_at';
      $valuesSql[]  = "DATE_ADD(NOW(), INTERVAL 30 DAY)";
    }

    // ip
    if (!empty($cols['ip'])) {
      $insertCols[] = 'ip';
      $valuesSql[]  = '?';
      $params[]     = $ip;
    }

    // user_agent
    if (!empty($cols['user_agent'])) {
      $insertCols[] = 'user_agent';
      $valuesSql[]  = '?';
      $params[]     = $ua;
    }

    // device (اختياري)
    if (!empty($cols['device'])) {
      $insertCols[] = 'device';
      $valuesSql[]  = '?';
      $params[]     = 'web';
    }

    // revoked flags (اختياري)
    if (!empty($cols['is_revoked'])) {
      $insertCols[] = 'is_revoked';
      $valuesSql[]  = '0';
    } elseif (!empty($cols['revoked'])) {
      $insertCols[] = 'revoked';
      $valuesSql[]  = '0';
    }

    $sql = "INSERT INTO user_tokens (" . implode(',', $insertCols) . ") VALUES (" . implode(',', $valuesSql) . ")";
    $st = $conn->prepare($sql);
    $st->execute($params);
  } catch (Throwable $e) {
    // لا نوقف تسجيل الدخول لو فشل تخزين التوكن
  }
}

/* ================= تشغيل ================= */
require_once __DIR__ . '/../config.php';
if (!isset($conn) || !($conn instanceof PDO)) {
  respond(false, 'اتصال قاعدة البيانات غير متوفر.', [], 500);
}

try {
  $input = read_json_input();

  // قد يكون بريد أو رقم حساب أو username
  $identifier = trim((string)($input['email'] ?? ''));
  $password   = trim((string)($input['password'] ?? ''));

  $recaptchaToken =
    trim((string)($input['recaptcha_token'] ?? '')) ?:
    trim((string)($input['g_recaptcha_response'] ?? '')) ?:
    trim((string)($input['g-recaptcha-response'] ?? ''));

  if ($identifier === '' || $password === '') {
    respond(false, 'يرجى إدخال بيانات الدخول كاملة.', [], 400);
  }

  if (strlen($password) < 6) {
    respond(false, 'كلمة المرور غير صحيحة.', [], 401);
  }

  // ✅ reCAPTCHA إلزامي
  if ($recaptchaToken === '') {
    respond(false, 'التحقق مطلوب. تأكد من عدم حجب reCAPTCHA ثم أعد المحاولة.', [
      'captcha' => false,
      'captcha_required' => true
    ], 403);
  }

  $remoteIp = $_SERVER['REMOTE_ADDR'] ?? '';
  $cap = verify_recaptcha($recaptchaToken, $remoteIp, 'login');
  if (!$cap['ok']) {
    $extra = ['captcha' => false];
    if ($DEBUG_MODE) $extra['captcha_reason'] = $cap['reason'] ?? 'unknown';
    respond(false, 'تعذّر تنفيذ reCAPTCHA. تأكد من عدم حجبها (AdBlock) ثم أعد المحاولة.', $extra, 403);
  }

  // تحديد نوع المعرف
  $isEmail = (bool)filter_var($identifier, FILTER_VALIDATE_EMAIL);
  $isDigits = ctype_digit($identifier);

  // جلب المستخدم
  if ($isEmail) {
    $st = $conn->prepare("SELECT id, email, username, account_number, password, is_active FROM users WHERE email = ? LIMIT 1");
    $st->execute([strtolower($identifier)]);
  } elseif ($isDigits) {
    $st = $conn->prepare("SELECT id, email, username, account_number, password, is_active FROM users WHERE account_number = ? LIMIT 1");
    $st->execute([$identifier]);
  } else {
    $st = $conn->prepare("SELECT id, email, username, account_number, password, is_active FROM users WHERE username = ? LIMIT 1");
    $st->execute([$identifier]);
  }

  $user = $st->fetch(PDO::FETCH_ASSOC);
  if (!$user) {
    respond(false, 'بيانات الدخول غير صحيحة.', [], 401);
  }

  // لو غير مفعل
  if ((int)$user['is_active'] !== 1) {
    respond(false, 'الحساب غير مُفعّل. سيتم تحويلك لتأكيد البريد.', [
      'needs_verification' => true,
      'email' => (string)$user['email']
    ], 200);
  }

  // تحقق كلمة المرور
  $hash = (string)($user['password'] ?? '');
  if ($hash === '' || !password_verify($password, $hash)) {
    respond(false, 'بيانات الدخول غير صحيحة.', [], 401);
  }

  // نجاح
  $userId = (int)$user['id'];
  $_SESSION['user_id'] = $userId;

  $token = bin2hex(random_bytes(32));
  $ua = substr((string)($_SERVER['HTTP_USER_AGENT'] ?? ''), 0, 255);
  store_user_token($conn, $userId, $token, $remoteIp, $ua);

  respond(true, 'تم تسجيل الدخول بنجاح', [
    'user_id'    => $userId,
    'user_token' => $token,
    'email'      => (string)$user['email'],
    'username'   => (string)($user['username'] ?? ''),
    'account_number' => (string)($user['account_number'] ?? ''),
    'email_verified' => 1
  ], 200);

} catch (PDOException $e) {
  log_error($e);
  $msg = 'خطأ في الخادم';
  if ($DEBUG_MODE) $msg .= ' - ' . $e->getMessage();
  respond(false, $msg, [], 500);

} catch (Throwable $e) {
  log_error($e);
  $msg = 'خطأ في الخادم';
  if ($DEBUG_MODE) $msg .= ' - ' . $e->getMessage();
  respond(false, $msg, [], 500);
}
