<?php
session_start();

/**
 * get_register.php (PDO)
 * ✅ التحقق من تكرار البريد يتم من جدول users فقط
 * - يمنع إعادة التسجيل إذا البريد موجود في users (سواء مفعل أو غير مفعل)
 * - عند تسجيل جديد: إنشاء حساب غير مفعل is_active=0
 * - إنشاء/إرسال كود تفعيل عبر جدول email_code (صلاحية 60 ثانية)
 * - لا يوجد action=resend_code هنا (إعادة الإرسال تكون عبر email_request.php)
 *
 * ملاحظة مهمة: يُفضّل جداً وجود UNIQUE على users.email لضمان منع التكرار نهائياً.
 */

// ================= إعدادات CORS =================
$frontendOrigin = 'https://yemencash.net';

$reqOrigin = $_SERVER['HTTP_ORIGIN'] ?? '';
if ($reqOrigin === $frontendOrigin) {
  header("Access-Control-Allow-Origin: {$frontendOrigin}");
  header("Vary: Origin");
  header("Access-Control-Allow-Credentials: true");
}
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Max-Age: 600");
header("Content-Type: application/json; charset=UTF-8");

// وضع تطوير
$DEBUG_MODE = true;
if ($DEBUG_MODE) { ini_set('display_errors', 1); error_reporting(E_ALL); }
else { ini_set('display_errors', 0); error_reporting(0); }

// Preflight
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') {
  http_response_code(204);
  exit;
}

// POST فقط
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success' => false, 'message' => 'طريقة الطلب غير مسموحة. استخدم POST فقط.'], JSON_UNESCAPED_UNICODE);
  exit;
}

// ================= أدوات مساعدة =================
function respond(bool $success, string $message, array $extra = [], int $status = 200): void {
  http_response_code($status);
  echo json_encode(
    array_merge(['success' => $success, 'message' => $message], $extra),
    JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES
  );
  exit;
}

function read_json_input(): array {
  $raw = file_get_contents('php://input') ?: '';
  $data = json_decode($raw, true);
  if (!is_array($data)) {
    respond(false, 'البيانات المرسلة غير صالحة', [], 400);
  }
  return $data;
}

function log_error(Throwable $e): void {
  $dir = __DIR__ . '/../../logs';
  if (!is_dir($dir)) @mkdir($dir, 0755, true);
  $file = $dir . '/api_errors_' . date('Y-m-d') . '.log';
  $line = "[" . date('Y-m-d H:i:s') . "] " . $e->getMessage() . " | " . $e->getFile() . ":" . $e->getLine() . PHP_EOL;
  @file_put_contents($file, $line, FILE_APPEND | LOCK_EX);
}

function sendMailCode(string $email, string $name, string $code, string $expiresAt): bool {
  $subject = "رمز تفعيل بريدك في YemenCash";
  $body = "مرحباً " . ($name !== '' ? $name : 'عزيزنا المستخدم') . "\n\n"
        . "رمز التفعيل الخاص بحسابك هو:\n"
        . $code . "\n\n"
        . "صالح حتى: " . $expiresAt . " (لمدة دقيقة واحدة).\n\n"
        . "فريق YemenCash";

  $headers  = "MIME-Version: 1.0\r\n";
  $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
  $headers .= "From: YemenCash <no-reply@yemencash.net>\r\n";

  return (bool) @mail($email, $subject, $body, $headers);
}

function sendVerificationCode(PDO $conn, int $userId, string $email, string $name): array {
  $ttlSeconds = 60;

  // توليد كود 6 أرقام
  $code = (string) random_int(100000, 999999);
  $expiresAt = date('Y-m-d H:i:s', time() + $ttlSeconds);

  // إغلاق الأكواد السابقة غير المستخدمة
  $conn->prepare("UPDATE email_code SET used = 1 WHERE user_id = ? AND used = 0")
       ->execute([$userId]);

  // إدخال كود جديد
  $ins = $conn->prepare("
    INSERT INTO email_code (email, user_id, code, used, expires_at, created_at)
    VALUES (?, ?, ?, 0, ?, NOW())
  ");
  $ins->execute([$email, $userId, $code, $expiresAt]);

  // إرسال البريد
  $sent = sendMailCode($email, $name, $code, $expiresAt);
  if (!$sent) {
    return ['success' => false, 'message' => 'تم إنشاء رمز التفعيل ولكن تعذّر إرسال البريد. حاول لاحقاً.'];
  }

  return [
    'success'    => true,
    'message'    => 'تم إرسال رمز التفعيل إلى بريدك الإلكتروني.',
    'expires_at' => $expiresAt,
    'ttl'        => $ttlSeconds
  ];
}

// ================= تشغيل =================
require_once __DIR__ . '/../config.php';
if (!isset($conn) || !($conn instanceof PDO)) {
  respond(false, 'اتصال قاعدة البيانات غير متوفر.', [], 500);
}

try {
  $input = read_json_input();

  $full_name = trim((string)($input['full_name'] ?? '')); // اختياري
  $email     = strtolower(trim((string)($input['email'] ?? '')));
  $password  = trim((string)($input['password'] ?? ''));

  if ($email === '' || $password === '') {
    respond(false, 'بعض الحقول ناقصة', [], 400);
  }
  if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    respond(false, 'البريد الإلكتروني غير صالح', [], 400);
  }
  if (strlen($password) < 6) {
    respond(false, 'كلمة المرور يجب أن تكون 6 أحرف على الأقل', [], 400);
  }

  // ✅ التحقق من users فقط (اختياري قبل الإدخال لرسالة أوضح)
  $chk = $conn->prepare("SELECT id, is_active FROM users WHERE email = ? LIMIT 1");
  $chk->execute([$email]);
  $existing = $chk->fetch(PDO::FETCH_ASSOC);
  if ($existing) {
    respond(false, 'البريد مسجل مسبقاً. استخدم تسجيل الدخول.', [
      'exists'    => true,
      'is_active' => (int)$existing['is_active']
    ], 409);
  }

  $conn->beginTransaction();

  // إنشاء حساب جديد غير مفعل
  $ins = $conn->prepare("
    INSERT INTO users (name, email, password, is_active, created_at)
    VALUES (?, ?, ?, 0, NOW())
  ");
  $ins->execute([
    $full_name,
    $email,
    password_hash($password, PASSWORD_DEFAULT)
  ]);

  $userId = (int)$conn->lastInsertId();
  if ($userId <= 0) {
    $conn->rollBack();
    respond(false, 'تعذّر إنشاء الحساب.', [], 500);
  }

  // إرسال رمز التفعيل (وتخزينه)
  $result = sendVerificationCode($conn, $userId, $email, $full_name);
  if (!$result['success']) {
    $conn->rollBack();
    respond(false, $result['message'], [], 500);
  }

  $conn->commit();

  respond(true, $result['message'], [
    'user_id'    => $userId,
    'email'      => $email,
    'is_active'  => 0,
    'expires_at' => $result['expires_at'] ?? null,
    'ttl'        => $result['ttl'] ?? 60
  ]);

} catch (PDOException $e) {
  if ($conn instanceof PDO && $conn->inTransaction()) $conn->rollBack();

  // في حال كان لديك UNIQUE على users.email قد يأتي خطأ تكرار هنا أيضاً
  if (($e->getCode() ?? '') === '23000') {
    $email = isset($email) ? (string)$email : '';
    if ($email !== '') {
      $stmt = $conn->prepare("SELECT is_active FROM users WHERE email = ? LIMIT 1");
      $stmt->execute([$email]);
      $u = $stmt->fetch(PDO::FETCH_ASSOC);

      respond(false, 'البريد مسجل مسبقاً. استخدم تسجيل الدخول.', [
        'exists'    => true,
        'is_active' => isset($u['is_active']) ? (int)$u['is_active'] : 0
      ], 409);
    }

    respond(false, 'البريد مسجل مسبقاً. استخدم تسجيل الدخول.', ['exists' => true], 409);
  }

  log_error($e);
  $msg = 'خطأ في الخادم';
  if ($DEBUG_MODE) $msg .= ' - ' . $e->getMessage();
  respond(false, $msg, [], 500);

} catch (Throwable $e) {
  if ($conn instanceof PDO && $conn->inTransaction()) $conn->rollBack();
  log_error($e);
  $msg = 'خطأ في الخادم';
  if ($DEBUG_MODE) $msg .= ' - ' . $e->getMessage();
  respond(false, $msg, [], 500);
}
